# Marp Theme CSS Creation Guide

How to create custom themes based on the official Marpit theme CSS specification.

Official documentation: https://marpit.marp.app/theme-css

## Basic Theme Structure

### @theme Metadata (Required)

Theme CSS requires `@theme` metadata:

```css
/* @theme theme-name */
```

Without this metadata, it won't be recognized as a theme.

### Basic Theme CSS Example

```css
/* @theme my-theme */

section {
  background-color: #fff;
  color: #333;
  font-size: 24px;
  padding: 60px;
}

h1 {
  color: #1e40af;
  font-size: 48px;
}

h2 {
  color: #3b82f6;
  font-size: 36px;
}
```

## HTML Structure

HTML structure generated by Marp:

```html
<section>
  <h1>Slide Title</h1>
  <p>Content</p>
</section>
```

Each slide is generated as a `<section>` element.

## Slide Size

### Default Size

```css
section {
  width: 1280px;
  height: 720px;
}
```

16:9 ratio (1280x720) is the default.

### Custom Size Definition

```css
/* @theme my-theme */
/* @size 4:3 960px 720px */

section {
  width: 960px;
  height: 720px;
}
```

Custom sizes can be defined with `@size` metadata.

## Pagination

Styling page numbers:

```css
section::after {
  content: attr(data-marpit-pagination) ' / ' attr(data-marpit-pagination-total);
  position: absolute;
  right: 30px;
  bottom: 20px;
  font-size: 16px;
  color: #666;
}
```

**Available attributes**:
- `data-marpit-pagination` - Current page number
- `data-marpit-pagination-total` - Total page count

## Headers and Footers

### Header

```css
header {
  position: absolute;
  top: 30px;
  left: 60px;
  right: 60px;
  font-size: 18px;
  color: #666;
}
```

### Footer

```css
footer {
  position: absolute;
  bottom: 30px;
  left: 60px;
  right: 60px;
  font-size: 18px;
  color: #666;
}
```

## Theme Extension and Inheritance

### Loading with @import

```css
/* @theme my-extended-theme */

@import 'default';

section {
  background-color: #f0f0f0;
}
```

You can extend existing themes.

### Inheriting Named Themes with @import-theme

```css
/* @theme my-theme */

@import-theme 'default';

h1 {
  color: red;
}
```

Using `@import-theme` allows loading by theme name.

## Class-Based Variations

### lead Class (Title Slide)

```css
section.lead {
  display: flex;
  flex-direction: column;
  justify-content: center;
  align-items: center;
  text-align: center;
}

section.lead h1 {
  font-size: 64px;
}
```

### invert Class (Inverted Colors)

```css
section.invert {
  background-color: #000;
  color: #fff;
}

section.invert h1,
section.invert h2 {
  color: #fff;
}
```

## Scoped Style (Slide-Specific Styles)

Use `<style scoped>` in Markdown:

```markdown
<style scoped>
section {
  background-color: #e3f2fd;
}
</style>

# Only this slide has blue background
```

## Background Image Styling

Background images are processed automatically but can be adjusted with CSS:

```css
section[data-marpit-background-image] {
  background-size: cover;
  background-position: center;
}
```

## Math Formula Styling (Marp Core)

```css
.katex {
  font-size: 1.2em;
}

.katex-display {
  margin: 1em 0;
}
```

## List Styling

```css
ul, ol {
  margin: 0.5em 0;
  padding-left: 1.5em;
}

li {
  margin: 0.3em 0;
}

li::marker {
  color: #3b82f6;
}
```

## Table Styling

```css
table {
  border-collapse: collapse;
  width: 100%;
  margin: 1em 0;
}

th, td {
  border: 1px solid #ddd;
  padding: 0.5em 1em;
  text-align: left;
}

th {
  background-color: #f0f0f0;
  font-weight: bold;
}
```

## Code Block Styling

```css
pre {
  background-color: #f5f5f5;
  border-radius: 4px;
  padding: 1em;
  overflow-x: auto;
}

code {
  font-family: 'Courier New', monospace;
  font-size: 0.9em;
}

pre code {
  background-color: transparent;
  padding: 0;
}
```

## Responsive Design

```css
@media screen and (max-width: 1280px) {
  section {
    font-size: 20px;
  }

  h1 {
    font-size: 40px;
  }
}
```

## Practical Theme Examples

### Minimal Theme

```css
/* @theme minimal */

section {
  background-color: #ffffff;
  color: #333333;
  font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
  font-size: 24px;
  padding: 80px;
}

h1, h2, h3 {
  font-weight: 300;
  color: #000000;
}

h1 {
  font-size: 60px;
  margin-bottom: 0.5em;
}

h2 {
  font-size: 40px;
  margin-bottom: 0.5em;
}
```

### Dark Theme

```css
/* @theme dark */

section {
  background-color: #1a1a1a;
  color: #e0e0e0;
  font-size: 22px;
  padding: 60px;
}

h1, h2, h3 {
  color: #61dafb;
}

a {
  color: #61dafb;
}

code {
  background-color: #2d2d2d;
  color: #61dafb;
}
```

## Embedding Styles in Markdown

```markdown
---
marp: true
---

<style>
section {
  background-color: #f8f8f4;
  font-family: 'Noto Sans JP', sans-serif;
}

h1 {
  color: #4f86c6;
}
</style>

# Slide
```

This method allows applying custom styles without using a theme.

## Best Practices

1. **Ensure Contrast**: Maintain a contrast ratio of 4.5:1 or higher between background and text colors
2. **Appropriate Font Sizes**: Body text 22-24px, titles 40-60px
3. **Sufficient Spacing**: Recommended padding of 60px or more
4. **Fallback Fonts**: Specify system fonts for when web fonts fail to load
5. **Print Support**: Define print styles with `@media print`

## Official References

For details, refer to official documentation:
- **Theme CSS**: https://marpit.marp.app/theme-css
- **Marpit API**: https://marpit-api.marp.app/
- **Official Theme Implementation**: https://github.com/marp-team/marp-core/tree/main/themes
